/*********
  M5Camera-FPV Car
  走れ！モバイルバッテリー（青）
  SoftAP Ver.0.1 FS90R版
  PC・スマホのブラウザにFPVコントローラーを表示し操縦で
  by RoboFarm  https://www.robofarm.jp
*********/

//　ライブラリ読込
#include "esp_camera.h"
#include <WiFi.h>
#include "soc/rtc_cntl_reg.h"              //電圧低下検出無効化
#include "esp_http_server.h"
#include "SPIFFS.h"                        //SPIFFS

const char *path_root   = "/index.html";   //dataフォルダ内のHTMLファイルはこのファイル名で
#define BUFFER_SIZE 5120                   //HTMLファイル格納配列最大サイズ5kByte
uint8_t buf[BUFFER_SIZE];                  //HTMLファイル送出用バッファ設定

const int L_Servo = 4;                    //左サーボGPIOpin 及びPWMチャンネル
const int LS_PWMChan = 15;
const int R_Servo = 13;                   //右サーボGPIOpin　及びチャンネル
const int RS_PWMChan = 14;
const int F_Speed = 60;                //前進・後退時速度全速
const int M_Speed = 30;                //前進・後退時中速速度
const int T_Speed = 12;                //旋回時速度

int LS_Offset=0;
int RS_Offset=0;


const int LedPin = 14;                     //LEDピン　Stream時に転倒　※LOW点灯

//SoftAP設定
const char* ssid = "MB_CarAP_B01";       //SSID
const char* pass = "test0000";           //パスワード

const IPAddress ip(192, 168, 4, 1);             //IPアドレス固定（割当可能範囲はDHCPで確認）
const IPAddress subnet(255, 255, 255, 0);          //

#define PART_BOUNDARY "123456789000000000000987654321"

static const char* _STREAM_CONTENT_TYPE = "multipart/x-mixed-replace;boundary=" PART_BOUNDARY;
static const char* _STREAM_BOUNDARY = "\r\n--" PART_BOUNDARY "\r\n";
static const char* _STREAM_PART = "Content-Type: image/jpeg\r\nContent-Length: %u\r\n\r\n";

httpd_handle_t stream_httpd = NULL;
httpd_handle_t camera_httpd = NULL;
sensor_t *sensor = NULL;

//http割り込み処理
// "/"割り込み時
static esp_err_t index_handler(httpd_req_t *req) {
  httpd_resp_set_type(req, "text/html");
  return httpd_resp_send(req, (const char *)buf, BUFFER_SIZE);
}

// "/FSFW"割り込み→全速前進時
static esp_err_t cmdFSFW_handler(httpd_req_t *req) {
  httpd_resp_set_type(req, "text/html");
  LeftServo(F_Speed);
  RightServo(F_Speed);
  Serial.println("FS-ForWard");
  return httpd_resp_send(req, NULL, 0);
}

// "/MSFW"割り込み→半速前進時
static esp_err_t cmdMSFW_handler(httpd_req_t *req) {
  httpd_resp_set_type(req, "text/html");
  LeftServo(M_Speed);
  RightServo(M_Speed);
  Serial.println("MS-ForWard");
  return httpd_resp_send(req, NULL, 0);
}

// "/MSLT"割り込み→半速左旋回時
static esp_err_t cmdMSLT_handler(httpd_req_t *req) {
  httpd_resp_set_type(req, "text/html");
  LeftServo(-T_Speed);
  RightServo(T_Speed);
  Serial.println("MS-LeftTurn");
  return httpd_resp_send(req, NULL, 0);
}


// "/ST"割り込み→Stop時
static esp_err_t cmdSTOP_handler(httpd_req_t *req) {
  httpd_resp_set_type(req, "text/html");
  LeftServo(0);
  RightServo(0);
  Serial.println("STOP");
  return httpd_resp_send(req, NULL, 0);
}

// "/MSRT"割り込み→半速右旋回時
static esp_err_t cmdMSRT_handler(httpd_req_t *req) {
  httpd_resp_set_type(req, "text/html");
  LeftServo(T_Speed);
  RightServo(-T_Speed);
  Serial.println("MS-RightTurn");
  return httpd_resp_send(req, NULL, 0);
}

// "/MSBW"割り込み→半速後退時
static esp_err_t cmdMSBW_handler(httpd_req_t *req) {
  httpd_resp_set_type(req, "text/html");
  LeftServo(-M_Speed);
  RightServo(-M_Speed);
  Serial.println("MS-BackWard");
  return httpd_resp_send(req, NULL, 0);
}

// "/FSBW"割り込み→全速後退時
static esp_err_t cmdFSBW_handler(httpd_req_t *req) {
  httpd_resp_set_type(req, "text/html");
  LeftServo(-F_Speed);
  RightServo(-F_Speed);
  Serial.println("FS-BackWard");
  return httpd_resp_send(req, NULL, 0);
}


// "/STREAM"割り込み→ストリーミング動画リクエスト時
static esp_err_t stream_handler(httpd_req_t *req) {
  digitalWrite(LedPin, LOW);           //LED点灯
  camera_fb_t * fb = NULL;
  esp_err_t res = ESP_OK;
  size_t _jpg_buf_len = 0;
  uint8_t * _jpg_buf = NULL;
  char * part_buf[64];

  res = httpd_resp_set_type(req, _STREAM_CONTENT_TYPE);
  if (res != ESP_OK) {
    digitalWrite(LedPin, HIGH);
    return res;
  }

  while (true) {
    fb = esp_camera_fb_get();
    if (!fb) {
      Serial.println("Camera capture failed");
      res = ESP_FAIL;
    } else {
      if (fb->width > 400) {
        if (fb->format != PIXFORMAT_JPEG) {
          bool jpeg_converted = frame2jpg(fb, 80, &_jpg_buf, &_jpg_buf_len);
          esp_camera_fb_return(fb);
          fb = NULL;
          if (!jpeg_converted) {
            Serial.println("JPEG compression failed");
            res = ESP_FAIL;
          }
        } else {
          _jpg_buf_len = fb->len;
          _jpg_buf = fb->buf;
        }
      }
    }
    if (res == ESP_OK) {
      size_t hlen = snprintf((char *)part_buf, 64, _STREAM_PART, _jpg_buf_len);
      res = httpd_resp_send_chunk(req, (const char *)part_buf, hlen);
    }
    if (res == ESP_OK) {
      res = httpd_resp_send_chunk(req, (const char *)_jpg_buf, _jpg_buf_len);
    }
    if (res == ESP_OK) {
      res = httpd_resp_send_chunk(req, _STREAM_BOUNDARY, strlen(_STREAM_BOUNDARY));
    }
    if (fb) {
      esp_camera_fb_return(fb);
      fb = NULL;
      _jpg_buf = NULL;
    } else if (_jpg_buf) {
      free(_jpg_buf);
      _jpg_buf = NULL;
    }
    if (res != ESP_OK) {
      digitalWrite(LedPin, HIGH);
      break;
    }
  }
  digitalWrite(LedPin, HIGH);
  return res;
}
//Streaming処理終わり

//カメラサーバー処理定義
void startCameraServer() {
  httpd_config_t config = HTTPD_DEFAULT_CONFIG();
  config.server_port = 80;

  //index割込定義
  httpd_uri_t index_uri = {
    .uri       = "/",
    .method    = HTTP_GET,
    .handler   = index_handler,
    .user_ctx  = NULL
  };

  //高速ForWard割込定義
  httpd_uri_t cmdFSFW_uri = {
    .uri       = "/FSFW",
    .method    = HTTP_GET,
    .handler   = cmdFSFW_handler,
    .user_ctx  = NULL
  };

  //中速ForWard割込定義
  httpd_uri_t cmdMSFW_uri = {
    .uri       = "/MSFW",
    .method    = HTTP_GET,
    .handler   = cmdMSFW_handler,
    .user_ctx  = NULL
  };

  //中速LeftTurn割込定義
  httpd_uri_t cmdMSLT_uri = {
    .uri       = "/MSLT",
    .method    = HTTP_GET,
    .handler   = cmdMSLT_handler,
    .user_ctx  = NULL
  };

  //Stop割込定義
  httpd_uri_t cmdSTOP_uri = {
    .uri       = "/STOP",
    .method    = HTTP_GET,
    .handler   = cmdSTOP_handler,
    .user_ctx  = NULL
  };

  //中速RightTurn割込定義
  httpd_uri_t cmdMSRT_uri = {
    .uri       = "/MSRT",
    .method    = HTTP_GET,
    .handler   = cmdMSRT_handler,
    .user_ctx  = NULL
  };


  //中速BackWard割込定義
  httpd_uri_t cmdMSBW_uri = {
    .uri       = "/MSBW",
    .method    = HTTP_GET,
    .handler   = cmdMSBW_handler,
    .user_ctx  = NULL
  };

  //高速BackWard割込定義
  httpd_uri_t cmdFSBW_uri = {
    .uri       = "/FSBW",
    .method    = HTTP_GET,
    .handler   = cmdFSBW_handler,
    .user_ctx  = NULL
  };


  //Streaming割込定義
  httpd_uri_t stream_uri = {
    .uri       = "/stream",
    .method    = HTTP_GET,
    .handler   = stream_handler,
    .user_ctx  = NULL
  };

  //ポート番号80のURLハンドラー登録
  Serial.printf("Starting web server on port: '%d'\n", config.server_port);
  if (httpd_start(&camera_httpd, &config) == ESP_OK) {
    httpd_register_uri_handler(camera_httpd, &index_uri);
    httpd_register_uri_handler(camera_httpd, &cmdFSFW_uri);
    httpd_register_uri_handler(camera_httpd, &cmdMSFW_uri);
    httpd_register_uri_handler(camera_httpd, &cmdMSLT_uri);
    httpd_register_uri_handler(camera_httpd, &cmdSTOP_uri);
    httpd_register_uri_handler(camera_httpd, &cmdMSRT_uri);
    httpd_register_uri_handler(camera_httpd, &cmdMSBW_uri);
    httpd_register_uri_handler(camera_httpd, &cmdFSBW_uri);
  }

  //ポート番号81のURLハンドラー登録
  config.server_port += 1;
  config.ctrl_port += 1;
  Serial.printf("Starting stream server on port: '%d'\n", config.server_port);
  if (httpd_start(&stream_httpd, &config) == ESP_OK) {
    httpd_register_uri_handler(stream_httpd, &stream_uri);
  }
}

//カメラ設定
void camera_setting() {
  //sensor->set_exposure_ctrl(sensor, 1);
  //sensor->set_aec2(sensor, 1);
  //sensor->set_aec_value(sensor, 100);
  //sensor->set_special_effect(sensor, 5);
  //sensor->set_agc_gain(sensor, 30);
  sensor->set_hmirror(sensor, 1);		//水平ミラー
  sensor->set_vflip(sensor, 1);			//垂直フリップ
  //sensor->set_raw_gma(sensor, 1);
  //sensor->set_whitebal(sensor, 1);
  //sensor->set_awb_gain(sensor, 1);
  //sensor->set_gain_ctrl(sensor, 1);
  //sensor->set_lenc(sensor, 0);
  //sensor->set_dcw(sensor, 1);
  //sensor->set_bpc(sensor, 0);
  //sensor->set_wpc(sensor, 1);
}

//サーボ処理関数
void LeftServo(int ang) {
  ledcWrite(LS_PWMChan, map(constrain(ang+LS_Offset, -90, 90), 90, -90, 123, 26));
}

void RightServo(int ang) {
  ledcWrite(RS_PWMChan, map(constrain(ang+RS_Offset, -90, 90), -90, 90, 123, 26));
}

void setup() {
  WRITE_PERI_REG(RTC_CNTL_BROWN_OUT_REG, 0); //電圧低下検出無効化

  Serial.begin(115200);
  Serial.setDebugOutput(false);
  WiFi.softAP(ssid, pass);
  delay(100);
  WiFi.softAPConfig(ip, ip, subnet);
  IPAddress myIP = WiFi.softAPIP();

  camera_config_t config;
  config.ledc_channel = LEDC_CHANNEL_0;
  config.ledc_timer = LEDC_TIMER_0;
  config.pin_d0 = 32;
  config.pin_d1 = 35;
  config.pin_d2 = 34;
  config.pin_d3 = 5;
  config.pin_d4 = 39;
  config.pin_d5 = 18;
  config.pin_d6 = 36;
  config.pin_d7 = 19;
  config.pin_xclk = 27;
  config.pin_pclk = 21;
  config.pin_vsync = 25;
  config.pin_href = 26;
  config.pin_sscb_sda = 22;
  config.pin_sscb_scl = 23;
  config.pin_pwdn = -1;
  config.pin_reset = 15;
  config.xclk_freq_hz = 20000000;
  config.pixel_format = PIXFORMAT_JPEG;
  config.frame_size = FRAMESIZE_VGA;
  config.jpeg_quality = 12; //Jpeg画質　小さい方が高画質
  config.fb_count = 2;  //フレームバッファー

  ledcSetup(LS_PWMChan, 50, 10);
  ledcAttachPin(L_Servo, LS_PWMChan); //左サーボ
  ledcSetup(RS_PWMChan, 50, 10);
  ledcAttachPin(R_Servo, RS_PWMChan); //右サーボ
  pinMode(LedPin, OUTPUT);

  //Servotest
  LeftServo(30);
  RightServo(30);
  delay(500);
  LeftServo(0);
  RightServo(0);

  // Camera 初期化
  esp_err_t err = esp_camera_init(&config);
  if (err != ESP_OK) {
    Serial.printf("Camera init failed with error 0x%x", err);
    return;
  }
  sensor = esp_camera_sensor_get();
  camera_setting();

  SPIFFS.begin();                                  //
  File htmlFile = SPIFFS.open(path_root, "r");     //
  size_t size = htmlFile.size();                  //
  htmlFile.read(buf, size);                        //
  htmlFile.close();                                //

  Serial.print("SSID: ");
  Serial.println(ssid);
  Serial.print("Camera Stream Ready! Go to: http://");
  Serial.println(myIP);

  // Start streaming web server
  startCameraServer();
}

void loop() {
  delay(1);
}
